/*
 * linux/arch/arm/mach-jasper/quasar.c
 * for JASPER / EM8550 with QUASAR 4
 * Created 09/19/2002 Ho Lee
 * Copyright 2002 Sigma Designs, Inc
 */

#ifdef __KERNEL__

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>

#include <asm/arch/quasar.h>

#else	// for boot loader

#include "quasar.h"

#define EXPORT_SYMBOL(x)
#define __init

#endif

void quasar_init(void);
void quasar_init_risc(void);
void quasar_init_2MB_SDRAM(void);
void quasar_init_4MB_SDRAM(void);
void quasar_init_2MB_EDORAM(void);

void quasar_lbc_init(void);
unsigned int quasar_lbc_readl(unsigned int addr);
void quasar_lbc_writel(unsigned int addr, unsigned int data);

void quasar_init_irq(void);

EXPORT_SYMBOL(quasar_lbc_readl);
EXPORT_SYMBOL(quasar_lbc_writel);

EXPORT_SYMBOL(quasar_enable_irq);
EXPORT_SYMBOL(quasar_disable_irq);
EXPORT_SYMBOL(quasar_irq_enabled);
EXPORT_SYMBOL(quasar_irq_pending);
EXPORT_SYMBOL(quasar_clear_irq);

int getLoaderSetCPUSpeed(void)
{
	int PLL_VALUE = 0, DIVIDER = 0, MULTIPLIER = 0, CPU_SPEED = 0;

	PLL_VALUE = *(volatile int*) (0x00600000 + ((0x1C04) << 2));
	DIVIDER = (PLL_VALUE >> 8) & 0xff;
	MULTIPLIER = (PLL_VALUE >> 2) & 0x3f;
	CPU_SPEED = (27*(MULTIPLIER+2))/((DIVIDER+2)*2);
	printk("Current Jasper CPU Speed %dMhz\n", CPU_SPEED);
	return CPU_SPEED;
}

void __init quasar_init(void)
{
	int temp = 0;
	int DIV = 1;	// Default to 198Mhz
	int MULT = 42;	// Default to 198Mhz

#ifdef ARCH_JASPER_108MHZ
	MULT = 22;
#endif
#ifdef ARCH_JASPER_117MHZ
	MULT = 24;
#endif
#ifdef ARCH_JASPER_126MHZ
	MULT = 26;
#endif
#ifdef ARCH_JASPER_135MHZ
	MULT = 28;
#endif
#ifdef ARCH_JASPER_144MHZ
	MULT = 30;
#endif
#ifdef ARCH_JASPER_153MHZ
	MULT = 32;
#endif
#ifdef ARCH_JASPER_162MHZ
	MULT = 34;
#endif
#ifdef ARCH_JASPER_171MHZ
	MULT = 36;
#endif
#ifdef ARCH_JASPER_180MHZ
	MULT = 38;
#endif
#ifdef ARCH_JASPER_189MHZ
	MULT = 40;
#endif
#ifdef ARCH_JASPER_198MHZ
	MULT = 42;
#endif
#ifdef ARCH_JASPER_202MHZ
	MULT = 43;
#endif
#ifdef ARCH_JASPER_207MHZ
#warning !!! Check memory chips - <70ns RAM is required !!!
	MULT = 44;
#endif

	quasar_init_risc();
	quasar_init_4MB_SDRAM();

	quasar_writel_reg(Q4_RISC_reset_run, 0x00);			// RISC unreset, stop
	quasar_writel_reg(Q4_MISC_reset1, 0x1818);
	quasar_writel_reg(Q4_MISC_reset1, 0x1800);
		
	quasar_lbc_init();

	getLoaderSetCPUSpeed();

	// Set CPU Speed
	// (27*(PLLREG_MULT+2)/(PLLREG_DIV+2)*2 Mhz)
	// fx: PLLREG_MULT = 42, PLLREG_DIV = 1, CPU Speed = 198Mhz
	temp = temp & 0xF000;
	temp = temp | 0x02;
	temp = temp | (DIV  << 8 );
	temp = temp | (MULT << 2 );
	*(volatile int *)(0x00600000 + (0x1C04 << 2)) = temp & 0x7fff;
	printk("Jasper CPU is now running @ %dMhz\n", (27*(MULT + 2))/((DIV+2)*2));
}

void __init quasar_init_risc(void)
{
	unsigned int temp;
		
	// QUASAR4 Enable
	quasar_writel_reg(0x1000 + RBUS_reset_run, 0x00);

	// Unreset RST0-7
	quasar_writel_reg(0x1000 + RISC_reset0, 0xffff);
	quasar_writel_reg(0x1000 + RISC_reset0, 0xff00);

	// Unreset RST8-15
	quasar_writel_reg(0x1000 + RISC_reset1, 0xffff);
	quasar_writel_reg(0x1000 + RISC_reset1, 0xff00);

	// Initialize SDRAM
	quasar_writel_reg(0x1000 + DRAM_config, (DRSIZE << 3) | TIMING | (REFRESH_PEROID << 5));

	// Program FIFO Size
	quasar_writel_reg(0x1000 + DRAM_fifosize0, 0x5555);
	quasar_writel_reg(0x1000 + DRAM_fifosize1, 0x055a);
	quasar_writel_reg(0x1000 + DRAM_fifosize2, 0x0080);

	// Unreset DRAM controller by clearing bit 15
	temp = quasar_readl_reg(0x1000 + DRAM_pllcontrol);
	quasar_writel_reg(0x1000 + DRAM_pllcontrol, temp & 0x7fff);
}

void __init quasar_init_2MB_SDRAM(void)
{
	quasar_writel_reg(0x1000 + DRAM_config, 0x40);		// 1 * SDRAM 2Mx16
}

void __init quasar_init_4MB_SDRAM(void)
{
	quasar_writel_reg(0x1000 + DRAM_config, 0x58);		// 1 * SDRAM 4Mx16
}

void __init quasar_init_2MB_EDORAM(void)
{
	quasar_writel_reg(0x1000 + DRAM_config, 0x40);		// 1 * EDO RAM 512kx32
}

//
// QUASAR Local Bus Controller
//

void __init quasar_lbc_init(void)
{
	quasar_writel(QUASAR_LBC_CONFIG0, 0x0002);			// set Streammachine if; MSB first for rd/wr; no auto incr
	quasar_writel(QUASAR_LBC_CONFIG1, 0x0000);			// prim. device; HPGIO input
}

unsigned int quasar_lbc_readl(unsigned int addr)
{
	unsigned int status, data;

	quasar_writel(QUASAR_LBC_READ_ADDR, addr);
	
	while ((status = quasar_readl(QUASAR_LBC_STATUS)) & 0x01)
		;
		
	data = quasar_readl(QUASAR_LBC_READ_DATA);

	return data;
}

void quasar_lbc_writel(unsigned int addr, unsigned int data)
{
	unsigned int status;

	quasar_writel(QUASAR_LBC_WRITE_ADDR, addr);
	quasar_writel(QUASAR_LBC_WRITE_DATA, data);
	
	while ((status = quasar_readl(QUASAR_LBC_STATUS)) & 0x02)
		;
}

//
// Interrupt handling
//

void __init quasar_init_irq(void)
{
	unsigned irqstat;

	// QUASAR LBC interrupt controller 
   
	irqstat = quasar_readl(QUASAR_LBC_INTERRUPT);
	irqstat &= 0x000f; 										// disable all interrupt, rising edge
															// clear all pending IRQ
	quasar_writel(QUASAR_LBC_INTERRUPT, irqstat);

	// HOST interrupt controller : enable Q2H_LOC interrupt
	
	irqstat = __raw_readl(JASPER_INT_CONTROLLER_BASE + INT_INTEN);
	irqstat |= Q2H_LOC_INT;
	__raw_writel(irqstat, JASPER_INT_CONTROLLER_BASE + INT_INTEN);
}

// enable specified IRQ from LBC
// irq : 0 - 3
// falling_edge : 0 = rising edge, 1 = falling edge
// return previous status of enable bit
int quasar_enable_irq(int irq, int falling_edge)
{
	unsigned int irqstat;
	int saved_enable;

	if (irq < 0 || irq > 3)
		return -1;
   
	irqstat = quasar_readl(QUASAR_LBC_INTERRUPT);
	saved_enable = (irqstat & (irq << 4)) ? 1 : 0;
	irqstat |= (1 << (4 + irq));							// set enable bit
	if (falling_edge)
		irqstat |= (1 << (8 + irq));						// set polarity bit : falling edge
	else 
		irqstat &= ~(1 << (8 + irq));						// clear polarity bit : rising edge
	irqstat &= ~0x000f;										// preserve IRQ pending
	quasar_writel(QUASAR_LBC_INTERRUPT, irqstat);

	return saved_enable;
}

int quasar_disable_irq(int irq)
{
	unsigned int irqstat;
	int saved_enable;

	if (irq < 0 || irq > 3)
		return -1;
   
	irqstat = quasar_readl(QUASAR_LBC_INTERRUPT);
	saved_enable = (irqstat & (irq << 4)) ? 1 : 0;
	irqstat &= ~(1 << (4 + irq));							// set enable bit
	irqstat &= ~0x000f;										// preserve IRQ pending
	quasar_writel(QUASAR_LBC_INTERRUPT, irqstat);

	return saved_enable;
}

int quasar_irq_enabled(int irq)
{
	unsigned int irqstat;

	if (irq < 0 || irq > 3)
		return -1;
   
	irqstat = quasar_readl(QUASAR_LBC_INTERRUPT);

	return (irqstat & (1 << (4 + irq))) ? 1 : 0;
}

// return if the specified IRQ is pending
int quasar_irq_pending(int irq)
{
	unsigned int irqstat;

	if (irq < 0 || irq > 3)
		return -1;
   
	irqstat = quasar_readl(QUASAR_LBC_INTERRUPT);
	return (irqstat & (1 << irq)) ? 1 : 0;
}

// clear the specified IRQ pending bit
void quasar_clear_irq(int irq)
{
	unsigned int irqstat;

	if (irq < 0 || irq > 3)
		return;
   
	irqstat = quasar_readl(QUASAR_LBC_INTERRUPT);

	if (irqstat & (1 << irq)) {
		irqstat &= ~0x000f;									// preserve IRQ pending
		irqstat |= (1 << irq);
		quasar_writel(QUASAR_LBC_INTERRUPT, irqstat);
	}
}

//
// DMA Handling
//

// addr : address of memory where the data will be written
// counter : number of bytes
void quasar_setup_q2h_dma(int master_enable, unsigned int addr, unsigned int counter)
{
	quasar_writel_reg(Q4_Q2H_master_enable, master_enable ? 1 : 0);
	quasar_writel_reg(Q4_Q2H_hostmem_addr_lo, addr & 0x0000ffff);
	quasar_writel_reg(Q4_Q2H_hostmem_addr_hi, (addr & 0xffff0000) >> 16);
	quasar_writel_reg(Q4_Q2H_byte_counter, counter);
}

// addr : address to read
// counter : number of bytes
void quasar_setup_lbc_readfifo(int fifo, unsigned int addr, unsigned int counter)
{
	quasar_writel_reg(Q4_LBC_config0, 0x1006);
	quasar_writel_reg(Q4_LBC_read_fifo0_access + (fifo * 2), addr);
	quasar_writel_reg(Q4_LBC_read_fifo0_cnt + (fifo * 2), counter);
}

void quasar_cleanup_lbc_fifo(void)
{
	quasar_writel_reg(Q4_LBC_config0, 0x0002);
}

//
// Miscellaneous
//

void quasar_flush_cache_all(void)
{
	__asm__(
		"mcr	p15, 0, %0, c7, c10, 0\n"		// write-back data cache
		"mcr	p15, 0, %0, c7, c6, 0\n"		// flush data cache
		"mcr	p15, 0, %0, c7, c5, 0\n"		// flush instruction cache
		: : "r" (0)
	);
}

void quasar_flush_cache_data(void)
{
	__asm__(
		"mcr	p15, 0, %0, c7, c10, 0\n"		// write-back data cache
		"mcr	p15, 0, %0, c7, c6, 0\n"		// flush data cache
		: : "r" (0)
	);
}

void quasar_flush_cache_data_region(unsigned int from, unsigned int to)
{
	from &= ~0x0f;
	to &= ~0x0f;
		
	for (; from <= to; from += 0x10) {
		__asm__ __volatile__ (
			"mcr	p15, 0, %0, c7, c10, 1\n"
			"mcr	p15, 0, %0, c7, c6, 1\n"
			: : "r" (from)
		);
	}
}

