/*
 *  linux/drivers/char/serial_jasper.c
 *
 *  Driver for JASPER serial ports
 *
 *  Based on drivers/char/serial.c, by Linus Torvalds, Theodore Ts'o.
 *
 *  Copyright 1999 ARM Limited
 *  Copyright (C) 2000 Deep Blue Solutions Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  $Id: serial_jasper.c,v 1.4 2001/07/17 20:34:27 rmk Exp $
 */

/*
 * This is a generic driver for JASPER and MAMBO serial ports.
 * Based on serial_amba.c
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/circ_buf.h>
#include <linux/serial.h>
#include <linux/console.h>
#include <linux/sysrq.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/bitops.h>

#if defined(CONFIG_SERIAL_JASPER_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/serial_core.h>

#include <asm/hardware/serial_jasper.h>

#define SERIAL_JASPER_MAJOR	204
#define SERIAL_JASPER_MINOR	16
#define SERIAL_JASPER_NR	UART_NR

#define CALLOUT_JASPER_NAME	"cuaam"
#define CALLOUT_JASPER_MAJOR	205
#define CALLOUT_JASPER_MINOR	16
#define CALLOUT_JASPER_NR	UART_NR

static struct tty_driver normal, callout;
static struct tty_struct *jasper_table[UART_NR];
static struct termios *jasper_termios[UART_NR], *jasper_termios_locked[UART_NR];
#ifdef SUPPORT_SYSRQ
static struct console jasper_console;
#endif

#define JASPER_ISR_PASS_LIMIT	256

/*
 * Access macros for the JASPER UARTs
 */

#define UART_GET_IER(p)		inl((p)->membase + JASPER_UART_IER)
#define UART_PUT_IER(p,c)	outl((c), (p)->membase + JASPER_UART_IER)
#define UART_GET_MSR(p)		inl((p)->membase + JASPER_UART_MSR)
#define UART_GET_LSR(p)		inl((p)->membase + JASPER_UART_LSR)
#define UART_GET_IIR(p)		inl((p)->membase + JASPER_UART_IIR)
#define UART_GET_RBR(p)		inl((p)->membase + JASPER_UART_RBR)
#define UART_PUT_TBR(p,c)	outl((c), (p)->membase + JASPER_UART_TBR)
#define UART_GET_LCR(p)		inl((p)->membase + JASPER_UART_LCR)
#define UART_PUT_LCR(p,c)	outl((c), (p)->membase + JASPER_UART_LCR)
#define UART_GET_FCR(p)		inl((p)->membase + JASPER_UART_FCR)
#define UART_PUT_FCR(p,c)	outl((c), (p)->membase + JASPER_UART_FCR)
#define UART_GET_MCR(p)		inl((p)->membase + JASPER_UART_MCR)
#define UART_PUT_MCR(p,c)	outl((c), (p)->membase + JASPER_UART_MCR)
#define UART_GET_CLKDIV(p)	inl((p)->membase + JASPER_UART_CLKDIV)
#define UART_PUT_CLKDIV(p,c)	outl((c), (p)->membase + JASPER_UART_CLKDIV)
#define UART_PUT_CLKSEL(p,c)    outl((c), (p)->membase + JASPER_UART_CLKSEL)

#define UART_GET_INT_STATUS(p)	(UART_GET_IIR(p) & 0xF)
#define UART_GET_CHAR(p)	UART_GET_RBR(p)
#define UART_PUT_CHAR(p, c)	UART_PUT_TBR(p, c)

/*
#define UART_GET_LCRL(p)	readb((p)->membase + JASPER_UARTLCR_L)
#define UART_PUT_LCRL(p,c)	writel((c), (p)->membase + JASPER_UARTLCR_L)
#define UART_GET_LCRM(p)	readb((p)->membase + JASPER_UARTLCR_M)
#define UART_PUT_LCRM(p,c)	writel((c), (p)->membase + JASPER_UARTLCR_M)
#define UART_GET_LCRH(p)	readb((p)->membase + JASPER_UARTLCR_H)
#define UART_PUT_LCRH(p,c)	writel((c), (p)->membase + JASPER_UARTLCR_H)
#define UART_RX_DATA(s)		(((s) & JASPER_UARTFR_RXFE) == 0)
#define UART_TX_READY(s)	(((s) & JASPER_UARTFR_TXFF) == 0)
#define UART_TX_EMPTY(p)	((UART_GET_FR(p) & JASPER_UARTFR_TMSK) == 0)
*/

#define UART_RX_READY(s)	((s)& JASPER_UART_LSR_DR)
#define UART_TX_READY(s)	((s)& JASPER_UART_LSR_THRE) // Tx holding reg is empty
#define UART_TX_EMPTY(s)	((s)& JASPER_UART_LSR_TEMT) // Tx shift reg is empty

#define UART_DUMMY_RSR_RX	256
#define UART_PORT_SIZE		64

#define RX_FIFO_TRIG_DEFAULT    8
#define TX_FIFO_TRIG_DEFAULT	8
#define RX_FIFO_TRIG(port)		((port)->unused[0])
#define TX_FIFO_TRIG(port)		((port)->unused[1])


/*
 * Our private driver data mappings.
 */
#define drv_old_status	driver_priv

static void jasperuart_stop_tx(struct uart_port *port, u_int from_tty)
{
	unsigned int ier;

	ier = UART_GET_IER(port);
	ier &= ~JASPER_UART_IER_ETBEI;
	UART_PUT_IER(port, ier);
}

static void jasperuart_start_tx(struct uart_port *port, u_int nonempty, u_int from_tty)
{
	if (nonempty) {
		unsigned int ier;

		ier = UART_GET_IER(port);
		UART_PUT_IER(port, 0);  // Need an edge to reenable transmit interrupt if it wasnt disabled
		ier |= JASPER_UART_IER_ETBEI;
		UART_PUT_IER(port, ier);
	}
}

static void jasperuart_stop_rx(struct uart_port *port)
{
	unsigned int ier;

	ier = UART_GET_IER(port);
	ier &= ~(JASPER_UART_IER_ERBFI);
	UART_PUT_IER(port, ier);
}

static void jasperuart_enable_ms(struct uart_port *port)
{
	unsigned int ier;

	ier = UART_GET_IER(port);
	ier |= JASPER_UART_IER_EDSSI;
	UART_PUT_IER(port, ier);
}

static void
#ifdef SUPPORT_SYSRQ
jasperuart_rx_chars(struct uart_info *info, struct pt_regs *regs)
#else
jasperuart_rx_chars(struct uart_info *info)
#endif
{
	struct tty_struct *tty = info->tty;
	unsigned int status, ch, max_count = 256;
	struct uart_port *port = info->port;

	status = UART_GET_LSR(port);
	while (UART_RX_READY(status) && max_count--) {
		if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
			tty->flip.tqueue.routine((void *)tty);
			if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
				printk(KERN_WARNING "TTY_DONT_FLIP set\n");
				return;
			}
		}

		ch = UART_GET_CHAR(port);

		*tty->flip.char_buf_ptr = ch;
		*tty->flip.flag_buf_ptr = TTY_NORMAL;
		port->icount.rx++;

		/*
		 * Note that the error handling code is
		 * out of the main execution path
		 */
		if (status & JASPER_UART_LSR_ANY) {
			if (status & JASPER_UART_LSR_BI) {
				status &= ~(JASPER_UART_LSR_FE | JASPER_UART_LSR_PE);
				port->icount.brk++;
				if (uart_handle_break(info, &jasper_console))
					goto ignore_char;
			} else if (status & JASPER_UART_LSR_PE)
				port->icount.parity++;
			else if (status & JASPER_UART_LSR_FE)
				port->icount.frame++;
			if (status & JASPER_UART_LSR_OE)
				port->icount.overrun++;

			status &= port->read_status_mask;

			if (status & JASPER_UART_LSR_BI)
				*tty->flip.flag_buf_ptr = TTY_BREAK;
			else if (status & JASPER_UART_LSR_PE)
				*tty->flip.flag_buf_ptr = TTY_PARITY;
			else if (status & JASPER_UART_LSR_FE)
				*tty->flip.flag_buf_ptr = TTY_FRAME;
		}

		if (uart_handle_sysrq_char(info, ch, regs))
			goto ignore_char;

		if ((status & port->ignore_status_mask) == 0) {
			tty->flip.flag_buf_ptr++;
			tty->flip.char_buf_ptr++;
			tty->flip.count++;
		}
		if ((status & JASPER_UART_LSR_OE) &&
		    tty->flip.count < TTY_FLIPBUF_SIZE) {
			/*
			 * Overrun is special, since it's reported
			 * immediately, and doesn't affect the current
			 * character
			 */
			*tty->flip.char_buf_ptr++ = 0;
			*tty->flip.flag_buf_ptr++ = TTY_OVERRUN;
			tty->flip.count++;
		}
	ignore_char:
		status = UART_GET_LSR(port);
	}
	tty_flip_buffer_push(tty);
	return;
}

/*
static char debug_str[9];
static char debug_ref[8]="FEEDME_W";
static char debug_pos=0;
*/

static void jasperuart_tx_chars(struct uart_info *info)
{
	struct uart_port *port = info->port;
	int count;

	if (port->x_char) {
		UART_PUT_CHAR(port, port->x_char);
		port->icount.tx++;
		port->x_char = 0;
		return;
	}
	if (info->xmit.head == info->xmit.tail
	    || info->tty->stopped
	    || info->tty->hw_stopped) {
		jasperuart_stop_tx(port, 0);
		return;
	}

	count = port->fifosize >> 1;
	do {
		UART_PUT_CHAR(port, info->xmit.buf[info->xmit.tail]);
		info->xmit.tail = (info->xmit.tail + 1) & (UART_XMIT_SIZE - 1);
		port->icount.tx++;
		if (info->xmit.head == info->xmit.tail)
			break;
	} while (--count > 0);

	if (CIRC_CNT(info->xmit.head, info->xmit.tail, UART_XMIT_SIZE) <
			WAKEUP_CHARS)
		uart_event(info, EVT_WRITE_WAKEUP);

	if (info->xmit.head == info->xmit.tail)
		jasperuart_stop_tx(info->port, 0);
}

static void jasperuart_modem_status(struct uart_info *info)
{
	struct uart_port *port = info->port;
	unsigned int status, delta;

	status = UART_GET_MSR(port);

	delta = status ^ info->drv_old_status;
	info->drv_old_status = status;

	if (!delta)
		return;

	if (delta & JASPER_UART_MSR_DCD)
		uart_handle_dcd_change(info, status & JASPER_UART_MSR_DCD);

	if (delta & JASPER_UART_MSR_DSR)
		port->icount.dsr++;

	if (delta & JASPER_UART_MSR_CTS)
		uart_handle_cts_change(info, status & JASPER_UART_MSR_CTS);

	wake_up_interruptible(&info->delta_msr_wait);
}

static void jasperuart_int(int irq, void *dev_id, struct pt_regs *regs)
{
	struct uart_info *info = dev_id;
	unsigned int status, pass_counter = JASPER_ISR_PASS_LIMIT;

	status = UART_GET_INT_STATUS(info->port);
	do {
		switch(status>>1)	{
		case UART_INT_RX_TIMEOUT:
		case UART_INT_RX_DATA:
#ifdef SUPPORT_SYSRQ
			jasperuart_rx_chars(info, regs);
#else
			jasperuart_rx_chars(info);
#endif
			break;
		case UART_INT_TX_EMPTY:
			jasperuart_tx_chars(info);
			break;
		case UART_INT_MODEM:
			jasperuart_modem_status(info);
			break;
		}
		if (pass_counter-- == 0)
			break;
		status = UART_GET_INT_STATUS(info->port);
	} while (!status & UART_INT_PENDING);
}

static u_int jasperuart_tx_empty(struct uart_port *port)
{
	return UART_TX_EMPTY(UART_GET_LSR(port)) ? TIOCSER_TEMT : 0;
}

static int jasperuart_get_mctrl(struct uart_port *port)
{
	unsigned int result = 0;
	unsigned int status;

	status = UART_GET_MSR(port);
	if (status & JASPER_UART_MSR_DCD)
		result |= TIOCM_CAR;
	if (status & JASPER_UART_MSR_DSR)
		result |= TIOCM_DSR;
	if (status & JASPER_UART_MSR_CTS)
		result |= TIOCM_CTS;

	/* Other modem lines not handled (RI,LOOP, OUT1 OUT2)*/

	return result;
}

static void jasperuart_set_mctrl(struct uart_port *port, u_int mctrl)
{
	u_int mcr = UART_GET_MCR(port);

	if (mctrl & TIOCM_RTS)
		mcr |= JASPER_UART_MCR_RTS;

	if (mctrl & TIOCM_DTR)
		mcr |= JASPER_UART_MCR_DTR;

	UART_PUT_MCR(port, mcr);
}

static void jasperuart_break_ctl(struct uart_port *port, int break_state)
{
	unsigned int lcr;

	lcr = UART_GET_LCR(port);
	if (break_state == -1)
		lcr |= JASPER_UART_LCR_SETBRK;
	else
		lcr &= ~JASPER_UART_LCR_SETBRK;
	UART_PUT_LCR(port, lcr);
}

static int jasperuart_startup(struct uart_port *port, struct uart_info *info)
{
	int retval;

	/*
	 * Allocate the IRQ
	 */
	retval = request_irq(port->irq, jasperuart_int, 0, "jasper", info);
	if (retval)
		return retval;

	/*
	 * initialise the old status of the modem signals
	 */
	info->drv_old_status = UART_GET_MSR(port);

	/*
	 * Finally, enable interrupts
	 * FIXME : Modem interrupt (EDSSI) 
	 */
	UART_PUT_IER(port, JASPER_UART_IER_ERBFI | JASPER_UART_IER_ETBEI | JASPER_UART_IER_ELSI );

	return 0;
}

static void jasperuart_shutdown(struct uart_port *port, struct uart_info *info)
{
	/*
	 * Free the interrupt
	 */
	free_irq(port->irq, info);

	/*
	 * disable all interrupts, disable the port
	 */
	UART_PUT_IER(port, 0);

	/* disable break condition and fifos */
	UART_PUT_LCR(port, UART_GET_LCR(port) &	~(JASPER_UART_LCR_SETBRK));
	UART_PUT_FCR(port, UART_GET_FCR(port) &	~(JASPER_UART_FCR_FIFOENA));
}

static void jasperuart_change_speed(struct uart_port *port, u_int cflag, u_int iflag, u_int quot)
{
	u_int lcr, fcr, old_ier;
	unsigned long flags;

#if DEBUG
	printk("jasperuart_set_cflag(0x%x) called\n", cflag);
#endif
	printk("jasperuart change speed %d\n",quot);
	/* byte size and parity */
	switch (cflag & CSIZE) {
	case CS5: lcr = JASPER_UART_LCR_WLEN_5; break;
	case CS6: lcr = JASPER_UART_LCR_WLEN_6; break;
	case CS7: lcr = JASPER_UART_LCR_WLEN_7; break;
	default:  lcr = JASPER_UART_LCR_WLEN_8; break; // CS8
	}
	if (cflag & CSTOPB)
		lcr |= JASPER_UART_LCR_STP2;
	if (cflag & PARENB) {
		lcr |= JASPER_UART_LCR_PEN;
		if (!(cflag & PARODD))
			lcr |= JASPER_UART_LCR_EPS;
	}
	fcr = JASPER_UART_FCR_TXRST | JASPER_UART_FCR_RXRST;
	if (port->fifosize > 1)
		fcr |= JASPER_UART_FCR_FIFOENA; // XXX: TX irq trigger

	UART_PUT_LCR(port, lcr);
	UART_PUT_FCR(port, fcr);
	
	port->read_status_mask = JASPER_UART_LSR_OE;
	if (iflag & INPCK)
		port->read_status_mask |= JASPER_UART_LSR_FE | JASPER_UART_LSR_PE;
	if (iflag & (BRKINT | PARMRK))
		port->read_status_mask |= JASPER_UART_LSR_BI;

	/*
	 * Characters to ignore
	 */
	port->ignore_status_mask = 0;
	if (iflag & IGNPAR)
		port->ignore_status_mask |= JASPER_UART_LSR_FE | JASPER_UART_LSR_PE;
	if (iflag & IGNBRK) {
		port->ignore_status_mask |= JASPER_UART_LSR_BI;
		/*
		 * If we're ignoring parity and break indicators,
		 * ignore overruns too (for real raw support).
		 */
		if (iflag & IGNPAR)
			port->ignore_status_mask |= JASPER_UART_LSR_OE;
	}

	/*
	 * Ignore all characters if CREAD is not set.
	 */
	if ((cflag & CREAD) == 0)
		port->ignore_status_mask |= UART_DUMMY_RSR_RX;

	/* first, disable everything */
	save_flags(flags); cli();
	old_ier = UART_GET_IER(port) & ~JASPER_UART_IER_EDSSI;

	if ((port->flags & ASYNC_HARDPPS_CD) ||
	    (cflag & CRTSCTS) || !(cflag & CLOCAL))
		old_ier |= JASPER_UART_IER_EDSSI;

	UART_PUT_IER(port, 0);

	/* Set baud rate */
#ifndef CONFIG_QUICKTURN_HACKS
	UART_PUT_CLKDIV(port, quot);
#else
#warning ***** QUICKTURN HACK *****  External UART Clock selected
	UART_PUT_CLKSEL(port, 1);
#endif
	UART_PUT_IER(port, old_ier);

	restore_flags(flags);
}

/*
 * Release the memory region(s) being used by 'port'
 */
static void jasperuart_release_port(struct uart_port *port)
{
	release_mem_region(port->mapbase, UART_PORT_SIZE);
}

/*
 * Request the memory region(s) being used by 'port'
 */
static int jasperuart_request_port(struct uart_port *port)
{
	return request_mem_region(port->mapbase, UART_PORT_SIZE, "serial_jasper")
			!= NULL ? 0 : -EBUSY;
}

/*
 * Configure/autoconfigure the port.
 */
static void jasperuart_config_port(struct uart_port *port, int flags)
{
	if (flags & UART_CONFIG_TYPE) {
		port->type = PORT_JASPER;
		jasperuart_request_port(port);
	}
}

/*
 * verify the new serial_struct (for TIOCSSERIAL).
 */
static int jasperuart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	int ret = 0;
	if (ser->type != PORT_UNKNOWN && ser->type != PORT_JASPER)
		ret = -EINVAL;
	if (ser->irq < 0 || ser->irq >= NR_IRQS)
		ret = -EINVAL;
	if (ser->baud_base < 9600)
		ret = -EINVAL;
	return ret;
}

static struct uart_ops jasper_pops = {
	tx_empty:	jasperuart_tx_empty,
	set_mctrl:	jasperuart_set_mctrl,
	get_mctrl:	jasperuart_get_mctrl,
	stop_tx:	jasperuart_stop_tx,
	start_tx:	jasperuart_start_tx,
	stop_rx:	jasperuart_stop_rx,
	enable_ms:	jasperuart_enable_ms,
	break_ctl:	jasperuart_break_ctl,
	startup:	jasperuart_startup,
	shutdown:	jasperuart_shutdown,
	change_speed:	jasperuart_change_speed,
	release_port:	jasperuart_release_port,
	request_port:	jasperuart_request_port,
	config_port:	jasperuart_config_port,
	verify_port:	jasperuart_verify_port,
};

static struct uart_port jasper_ports[UART_NR] = {
	{
		membase:	(void *)IO_ADDRESS(JASPER_UART0_BASE),
		mapbase:	JASPER_UART0_BASE,
		iotype:		SERIAL_IO_MEM,
		irq:		JASPER_UART0_IRQ,
		uartclk:	14745600,
		fifosize:	16,
		unused:		{ RX_FIFO_TRIG_DEFAULT, TX_FIFO_TRIG_DEFAULT},
		ops:		&jasper_pops,
		flags:		ASYNC_BOOT_AUTOCONF,
	},
	{
		membase:	(void *)IO_ADDRESS(JASPER_UART1_BASE),
		mapbase:	JASPER_UART1_BASE,
		iotype:		SERIAL_IO_MEM,
		irq:		JASPER_UART1_IRQ,
		uartclk:	14745600,
		fifosize:	16,
		unused:		{ RX_FIFO_TRIG_DEFAULT, TX_FIFO_TRIG_DEFAULT},
		ops:		&jasper_pops,
		flags:		ASYNC_BOOT_AUTOCONF,
	}
};

#ifdef CONFIG_SERIAL_JASPER_CONSOLE
#ifdef used_and_not_const_char_pointer
static int jasperuart_console_read(struct uart_port *port, char *s, u_int count)
{
	unsigned int status;
	int c;
#if DEBUG
	printk("jasperuart_console_read() called\n");
#endif

	c = 0;
	while (c < count) {
		status = UART_GET_LSR(port);
		if (UART_RX_READY(status)) {
			*s++ = UART_GET_CHAR(port);
			c++;
		} else {
			// nothing more to get, return
			return c;
		}
	}
	// return the count
	return c;
}
#endif

static void jasperuart_console_write(struct console *co, const char *s, u_int count)
{
	struct uart_port *port = jasper_ports + co->index;
	unsigned int status, old_ier;
	int i;

	/*
	 *	First save the IER then disable the interrupts
	 */
	old_ier = UART_GET_IER(port);
	UART_PUT_IER(port, 0);

	/*
	 *	Now, do each character
	 */
	for (i = 0; i < count; i++) {
		do {
			status = UART_GET_LSR(port);
		} while (!UART_TX_READY(status));
		UART_PUT_CHAR(port, s[i]);
		if (s[i] == '\n') {
			do {
				status = UART_GET_LSR(port);
			} while (!UART_TX_READY(status));
			UART_PUT_CHAR(port, '\r');
		}
	}

	/*
	 *	Finally, wait for transmitter to become empty
	 *	and restore the TCR
	 */
	do {
		status = UART_GET_LSR(port);
	} while (!UART_TX_EMPTY(status));
	UART_PUT_IER(port, old_ier);
}

static kdev_t jasperuart_console_device(struct console *co)
{
	return MKDEV(SERIAL_JASPER_MAJOR, SERIAL_JASPER_MINOR + co->index);
}

static int jasperuart_console_wait_key(struct console *co)
{
	struct uart_port *port = jasper_ports + co->index;
	unsigned int status;

	do {
		status = UART_GET_LSR(port);
	} while (!UART_RX_READY(status));
	return UART_GET_CHAR(port);
}

/*
static void __init
jasperuart_console_get_options(struct uart_port *port, int *baud, int *parity, int *bits)
{
	u_int lcr, quot;
	lcr = UART_GET_LCR(port);

	*parity = 'n';
	if (lcr & JASPER_UART_LCR_PEN) {
		if (lcr & JASPER_UART_LCR_EPS)
			*parity = 'e';
		else
			*parity = 'o';
	}
	*bits = (lcr & JASPER_UART_LCR_WLS) + 5;

	quot = UART_GET_CLKDIV(port);
	//FIXME : "+1" is here to avoid division by 0 errors,
	// need to check what REALLY happens when CLKDIV is set to 0, (disabled the UART or ?)
	*baud = port->uartclk / (16 * quot + 1);
}
*/

static int __init jasperuart_console_setup(struct console *co, char *options)
{
	struct uart_port *port;
	int baud = CONFIG_SERIAL_JASPER_SERIAL_BAUDRATE;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	/*
	 * Check whether an invalid uart number has been specified, and
	 * if so, search for the first available port that does have
	 * console support.
	 */
	printk("serial_jasper: setup_console @ %d\n",baud);

	port = uart_get_console(jasper_ports, UART_NR, co);

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);
//	else 
//		jasperuart_console_get_options(port, &baud, &parity, &bits);


	return uart_set_options(port, co, baud, parity, bits, flow);
}

static struct console jasper_console = {
	write:		jasperuart_console_write,
#ifdef used_and_not_const_char_pointer
	read:		jasperuart_console_read,
#endif
	device:		jasperuart_console_device,
	wait_key:	jasperuart_console_wait_key,
	setup:		jasperuart_console_setup,
	flags:		CON_PRINTBUFFER,
	index:		-1,
};

void __init jasperuart_clock_init(void)
{
	int i;
	
	for(i=0; i< UART_NR; i++) 
		jasper_ports[i].uartclk = __get_clock(2);
}

void __init jasperuart_console_init(void)
{
	jasperuart_clock_init();

	register_console(&jasper_console);
}


#define JASPER_CONSOLE	&jasper_console
#else
#define JASPER_CONSOLE	NULL
#endif

static struct uart_driver jasper_reg = {
	owner:			THIS_MODULE,
	normal_major:		SERIAL_JASPER_MAJOR,
#ifdef CONFIG_DEVFS_FS
	normal_name:		"ttyAM%d",
	callout_name:		"cuaam%d",
#else
	normal_name:		"ttyAM",
	callout_name:		"cuaam",
#endif
	normal_driver:		&normal,
	callout_major:		CALLOUT_JASPER_MAJOR,
	callout_driver:		&callout,
	table:			jasper_table,
	termios:		jasper_termios,
	termios_locked:		jasper_termios_locked,
	minor:			SERIAL_JASPER_MINOR,
	nr:			UART_NR,
	port:			jasper_ports,
	cons:			JASPER_CONSOLE,
};

static int __init jasperuart_init(void)
{
	jasperuart_clock_init();
	return uart_register_driver(&jasper_reg);
}

static void __exit jasperuart_exit(void)
{
	uart_unregister_driver(&jasper_reg);
}

module_init(jasperuart_init);
module_exit(jasperuart_exit);
