/*
 * Mapping of flash in Jasper - after remap
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/config.h>


#define WINDOW_ADDR 0
#define WINDOW_SIZE 4*1024*1024
#define BUSWIDTH 2

// JFTL : Jasper Flash TransLation (after remap)
// input: flash address - ouput: cpu address
#define JFTL(x) (void *)((x<0x2000)?(x+0x400000):(x))

static struct mtd_info *mymtd;

__u8 jaspermap_read8(struct map_info *map, unsigned long ofs)
{
	return __raw_readb(JFTL(ofs));
}

__u16 jaspermap_read16(struct map_info *map, unsigned long ofs)
{
	return __raw_readw(JFTL(ofs));
}

__u32 jaspermap_read32(struct map_info *map, unsigned long ofs)
{
	return __raw_readl(JFTL(ofs));
}

void jaspermap_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{       

	if(from<0x2000 && (from+len)>0x2000) { 
		// If we cross the 0x2000 boundary we have
		// to do split the trasnfert
		memcpy(to, JFTL(from), 0x2000-from);
		memcpy(to+0x2000-from, (void *)0x2000, len-0x2000+from);
	} else {
		memcpy(to, JFTL(from), len);
	}
}

void jaspermap_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	__raw_writeb(d, JFTL(adr));
	mb();
}

void jaspermap_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	__raw_writew(d, JFTL(adr));
	mb();
}

void jaspermap_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	__raw_writel(d, JFTL(adr));
	mb();
}

void jaspermap_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	if(to<0x2000 && (to+len)>0x2000) { 
		// If we cross the 0x2000 boundary we have
		// to do split the trasnfert
		memcpy(JFTL(to), from, 0x2000-to);
		memcpy((void *)0x2000, from+0x2000-to, len-0x2000+to);
	} else {
		memcpy(JFTL(to), from, len);
	}
}

struct map_info jaspermap_map = {
	name: "Jasper mapped flash",
	size: WINDOW_SIZE,
	buswidth: BUSWIDTH,
	read8: jaspermap_read8,
	read16: jaspermap_read16,
	read32: jaspermap_read32,
	copy_from: jaspermap_copy_from,
	write8: jaspermap_write8,
	write16: jaspermap_write16,
	write32: jaspermap_write32,
	copy_to: jaspermap_copy_to
};

int __init init_jaspermap(void)
{
       	printk(KERN_NOTICE "jaspermap flash device: %x at %x\n", WINDOW_SIZE, WINDOW_ADDR);

	mymtd = do_map_probe("cfi_probe", &jaspermap_map);
	if (mymtd) {
		mymtd->module = THIS_MODULE;

		add_mtd_device(mymtd);
		return 0;
	}

	return -ENXIO;
}

static void __exit cleanup_jaspermap(void)
{
	if (mymtd) {
		del_mtd_device(mymtd);
		map_destroy(mymtd);
	}
	if (jaspermap_map.map_priv_1) {
		jaspermap_map.map_priv_1 = 0;
	}
}

module_init(init_jaspermap);
module_exit(cleanup_jaspermap);


MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Jasper MTD map driver");
