/*
 * linux/include/asm-arm/arch-jasper/quasar.h
 * for supporting QUASAR 4
 * Created 09/19/2002 Ho Lee
 * Copyright 2002, Sigma Desings, Inc
 */

#ifndef __ASM_ARCH_QUASAR_H
#define __ASM_ARCH_QUASAR_H

#ifdef __KERNEL__

#include <asm/io.h>
#include <asm/hardware.h>

#include <asm/arch/jasper_const.h>
#include <asm/arch/jasper_q4.h>

#else	// for boot loader

#include "jasper.h"
#include "jasper_const.h"
#include "jasper_q4.h"

#include "io.h"

#endif

// Convention : 
//   target : 
//     quasar_xxx : QUASAR I/O (relative address to QUASAR_MAP_AREA)
//     quasar_xxx_reg : QUASAR I/O with register index : address = reg << 2 + QUASAR_MAP_AREA
//   operation unit : 
//     xxxb : 8 bits I/O
//     xxxw : 16 bits I/O
//     xxxl : 32 bits I/O

//
// Quasar I/O
//

static __inline__ unsigned char quasar_readb(unsigned int addr)
{
	return __raw_readb(QUASAR_MAP_AREA + addr);
}

static __inline__ unsigned short quasar_readw(unsigned int addr)
{
	return __raw_readw(QUASAR_MAP_AREA + addr);
}

static __inline__ unsigned int quasar_readl(unsigned int addr)
{
	return __raw_readl(QUASAR_MAP_AREA + addr);
}

static __inline__ unsigned int quasar_readl_reg(unsigned int reg)
{
	return __raw_readl(QUASAR_MAP_AREA + (reg << 2));
}

static __inline__ void quasar_writeb(unsigned int addr, unsigned int data)
{
	__raw_writeb(data, QUASAR_MAP_AREA + addr);
}

static __inline__ void quasar_writew(unsigned int addr, unsigned int data)
{
	__raw_writew(data, QUASAR_MAP_AREA + addr);
}

static __inline__ void quasar_writel(unsigned int addr, unsigned int data)
{
	__raw_writel(data, QUASAR_MAP_AREA + addr);
}

static __inline__ void quasar_writel_reg(unsigned int reg, unsigned int data)
{
	__raw_writel(data, QUASAR_MAP_AREA + (reg << 2));
}

//
// Other function prototypes
//

extern void quasar_init(void);
extern void quasar_init_irq(void);

extern unsigned int quasar_lbc_readl(unsigned int addr);
extern void quasar_lbc_writel(unsigned int addr, unsigned int data);

extern int quasar_enable_irq(int irq, int falling_edge);
extern int quasar_disable_irq(int irq);
extern int quasar_irq_enabled(int irq);
extern int quasar_irq_pending(int irq);
extern void quasar_clear_irq(int irq);

void quasar_setup_q2h_dma(int master_enable, unsigned int addr, unsigned int counter);
void quasar_setup_lbc_readfifo(int fifo, unsigned int addr, unsigned int counter);
void quasar_cleanup_lbc_fifo(void);

void quasar_flush_cache_all(void);
void quasar_flush_cache_data(void);
void quasar_flush_cache_data_region(unsigned int from, unsigned int to);

#endif
